#include "pch.h" // use stdafx.h in Visual Studio 2017 and earlier
#include "Converter.h"
#include <assert.h>
#include <stdio.h>

static unsigned int ConvertCan(char* buf, int size, const unsigned char* data);
static unsigned int ConvertLin(char* buf, int size, const unsigned char* data);
static unsigned int ConvertCxpi(char* buf, int size, const unsigned char* data);
static unsigned int ConvertPort(char* buf, int size, const unsigned char* data);
static unsigned int ConvertEnv(char* buf, int size, const unsigned char* data);
static unsigned int ConvertMarker(char* buf, int size, const unsigned char* data);

unsigned int Convert(char* buf
	, int type, int protocol, int channel, int status, int size, const unsigned char* data)
{
	// ^Cvo
	char lcode;
	switch (type)
	{
	case 0x3030: lcode = 'O'; break;	// 
	case 0x3031: lcode = 'I'; break;	// 
	case 0x3033: lcode = 'P'; break;	// |[gXe[^X
	case 0x3131: lcode = 'E'; break;	// Xe[^X
	case 0xFFFB: lcode = 'M'; break; // Marker
	default: return 0;
	}
	// vgRo͂
	const char* pcode;
	if (protocol == 0x04)			// CAN
		pcode = "CAN";
	else if (protocol == 0x0C)		// LIN
		pcode = "LIN";
	else if (protocol == 0x28)		// CXPI
		pcode = "CXPI";
	else if (type == 0x3033 && protocol == 0x00)	// P/I
		pcode = "P/I";
	else if (type == 0x3033 && protocol == 0x83)	// P/O
		pcode = "P/O";
	else if (type == 0x3131)			// Env
		pcode = "Env";
	else if (type == 0xFFFB)
		pcode = "Marker";
	else
		return 0;

	// `lo͂
	char ccode[3] = "-";
	switch (protocol)
	{
	case 0x04:	// CAN
	case 0x0C:	// LIN
	case 0x28:	// CXPI
	case 0x80:	// A/I
		sprintf(ccode, "%d", channel + 1); break;
	}

	// Xe[^Xo͂
	const char* scode;
	if (status == 0)
		scode = "OK";
	else
		scode = "NG";

	// wb_o͂
	char* buf2 = buf;
	buf2 += sprintf(buf2, ",%c,%-8s,%2s,%s", lcode, pcode, ccode, scode);
	// t[o͂
	if (protocol == 0x04)			// CAN
		buf2 += ConvertCan(buf2, size, data);
	else if (protocol == 0x0C)		// LIN
		buf2 += ConvertLin(buf2, size, data);
	else if (protocol == 0x28)		// CXPI
		buf2 += ConvertCxpi(buf2, size, data);
	else if (type == 0x3033)		// P/I,P/O
		buf2 += ConvertPort(buf2, size, data);
	else if (type == 0x3131)		// Env
		buf2 += ConvertEnv(buf2, size, data);
	else if (type == 0xFFFB)		// Marker
		buf2 += ConvertMarker(buf2, size, data);
	else
		assert(false);

	return buf2 - buf;
}

unsigned int ConvertCan(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	if (size--)
	{
		const int tbType[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };
		int type = tbType[*data >> 5];
		if ((type & 0x1) == 0)
		{
			// EXT=0/2/4/6, ID0
			buf2 += sprintf(buf2, ",%X,%X", type, *data++ & 0x7);
			if (size--)
			{
				// ID1
				buf2 += sprintf(buf2, ",%02X", *data++);
				if (size--)
				{
					// DLC
					buf2 += sprintf(buf2, ",%X", *data++ & 0xF);
					while (size--)
					{
						// D1..Dn
						buf2 += sprintf(buf2, ",%02X", *data++);
					}
				}
			}
		}
		else
		{
			// EXT=1/3/5/7, HD0
			buf2 += sprintf(buf2, ",%X,%02X", type, *data++ & 0x1F);
			if (size--)
			{
				// HD1
				buf2 += sprintf(buf2, ",%02X", *data++);
				if (size--)
				{
					// HD2
					buf2 += sprintf(buf2, ",%02X", *data++);
					if (size--)
					{
						// HD3
						buf2 += sprintf(buf2, ",%02X", *data++);
						if (size--)
						{
							// DLC
							buf2 += sprintf(buf2, ",%X", *data++ & 0xF);
							while (size--)
							{
								// D1..Dn
								buf2 += sprintf(buf2, ",%02X", *data++);
							}
						}
					}
				}
			}
		}
	}
	return buf2 - buf;
}

unsigned int ConvertLin(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	if (size--)
	{
		// M/S
		buf2 += sprintf(buf2, ",%X", *data++);
		if (size--)
		{
			// ID
			buf2 += sprintf(buf2, ",%02X", *data++);
			if (size--)
			{
				// P
				buf2 += sprintf(buf2, ",%X", *data++);
				while (size--)
				{
					// D1..Dn
					buf2 += sprintf(buf2, ",%02X", *data++);
				}
			}
		}
	}
	return buf2 - buf;
}

unsigned int ConvertCxpi(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	if (size--)
	{
		// TYPE
		buf2 += sprintf(buf2, ",%X", *data++ & 0x3);
		if (size--)
		{
			// P, ID
			buf2 += sprintf(buf2, ",%X,%02X", *data >> 7 & 0x1, *data & 0x7F);
			data++;
			if (size--)
			{
				// DLC, NM, CT
				buf2 += sprintf(buf2, ",%X,%X,%X", *data >> 4 & 0xF, *data >> 2 & 0x3, *data & 0x3);
				data++;
				while (size--)
				{
					// D1..Dn
					buf2 += sprintf(buf2, ",%02X", *data++);
				}
			}
		}
	}
	return buf2 - buf;
}

unsigned int ConvertPort(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	int num = *data++;
	DWORD c = 0;
	DWORD o = 0;
	for (int i = 0; i < num; ++i)
	{
		c |= *data++ << 8 * i;
		o |= *data++ << 8 * i;
	}
	for (int i = 0; i < 8 * num; ++i)
	{
		bool c1 = (c & (1 << i)) != 0;
		bool o1 = (o & (1 << i)) != 0;
		char v;
		if (c1 && o1 || !c1 && !o1)
			v = c1 ? '1' : '0';
		else
			v = c1 ? '+' : '-';
		buf2 += sprintf(buf2, ",P%d=%c", i + 1, v);
	}
	return buf2 - buf;
}

unsigned int ConvertEnv(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	*buf2++ = ',';
	while (*data)
		*buf2++ = *data++;
	data++;
	int len = *data++ << 8;
	len |= *data++;
	*buf2++ = ',';
	*buf2++ = '\"';
	while (len-- > 0)
		*buf2++ = *data++;
	*buf2++ = '\"';

	return buf2 - buf;
}

unsigned int ConvertMarker(char* buf, int size, const unsigned char* data)
{
	char* buf2 = buf;

	*buf2++ = ',';
	while (size-- > 0 && *data != '\0')
		*buf2++ = *data++;

	return buf2 - buf;
}
